<?php

/**
 *
 * Zoho Books API
 * Version: 2
 *
 * Author: Giuseppe Occhipinti - https://github.com/peppeocchi
 *
 * CHANGELOG v2
 * - extended parameters for invoices and credit notes and contacts
 *
 */

class ZohoBooks
{
	/**
	 * cUrl timeout
	 */
	private $timeout = 30;

	/**
	 * HTTP code of the cUrl request
	 */
	private $httpCode;

	/**
	 * Zoho Books API authentication
	 */
	private $authtoken;
	private $organizationId;

	/**
	 * Zoho Books API request limit management
	 */
	private $apiRequestsLimit = 150;
	private $apiRequestsCount;
	private $apiTimeLimit = 60;
	private $startTime;

	/**
	 * Zoho Books API urls request
	 */
	private $apiUrl = 'https://books.zoho.com/api/v3/';
	private $contactsUrl = 'contacts/';
	private $invoicesUrl = 'invoices/';
	private $estimatesUrl = 'estimates/';
	private $creditnotesUrl = 'creditnotes/';
	private $customerPaymentsUrl = 'customerpayments/';

	

	/**
	 * Init
	 *
	 * @param (string) Zoho Books authentication token
	 * @param (string) Zoho Books organization id
	 */
	public function __construct($authtoken, $organizationId)
	{
		$this->authtoken = $authtoken;
		$this->organizationId = $organizationId;
		$this->apiRequestsCount = 0;
		$this->startTime = time();
	}


	/**
	 * Get all contacts
	 *
	 * @return (string) json string || false
	 */
	public function allContacts($config = array())
	{
		$url = $this->apiUrl . $this->contactsUrl . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		if(isset($config['page'])) {
			$url .= '&page=' . $config['page'];
		}
		if(isset($config['sort_column'])) {
			$url .= '&sort_column=' . $config['sort_column'];
		}
		if(isset($config['phone_startswith'])) {
			$url .= '&phone_startswith=' . $config['phone_startswith'];
		}
		if(isset($config['phone_contains'])) {
			$url .= '&phone_contains=' . $config['phone_contains'];
		}
		
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$contacts = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $contacts : false;
	}

	/**
	 * Get contact details by ID
	 *
	 * @param (int) contact id
	 *
	 * @return (string) json string || false
	 */
	public function getContact($id)
	{
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $this->apiUrl . $this->contactsUrl . $id . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$contact = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $contact : false;
	}


	/**
	 * Get all invoices
	 *
	 * @param (date) date start
	 * @param (date) date end
	 *
	 * @return (string) json string || false
	 */
	public function allInvoices($config = array())
	{
		$url = $this->apiUrl . $this->invoicesUrl . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		if(isset($config['date_start']) && isset($config['date_end'])) {
			$url .= '&date_start=' . $config['date_start'] . '&date_end=' . $config['date_end'];
		}
		if(isset($config['invoice_number_startswith'])) {
			$url .= '&invoice_number_startswith=' . $config['invoice_number_startswith'];
		}
		if(isset($config['page'])) {
			$url .= '&page=' . $config['page'];
		}
		if(isset($config['item_description'])) {
			$url .= '&item_description=' . $config['item_description'];
		}
		if(isset($config['item_description_contains'])) {
			$url .= '&item_description_contains=' . $config['item_description_contains'];
		}
		if(isset($config['customer_id'])) {
			$url .= '&customer_id=' . $config['customer_id'];
		}
		if(isset($config['status'])) {
			$url .= '&status=' . $config['status'];
		}
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$invoices = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $invoices : false;
	}


	/**
	 * Get invoice
	 *
	 * @param (int) invoice id
	 *
	 * @return (string) json string || false
	 */
	public function getInvoice($id, $config = array())
	{
		$url = $this->apiUrl . $this->invoicesUrl . $id . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		if(isset($config['accept'])) {
			$url .= '&accept=' . $config['accept'];
		}
		if(isset($config['print'])) {
			$url .= '&print=' . $config['print'];
		}
		
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$invoice = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $invoice : false;
	}


	/**
	 * Get all estimates
	 *
	 * @param (date) date start
	 * @param (date) date end
	 *
	 * @return (string) json string || false
	 */
	public function allEstimates($config = array())
	{
		$url = $this->apiUrl . $this->estimatesUrl . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		if(isset($config['page'])) {
			$url .= '&page=' . $config['page'];
		}
		if(isset($config['customer_id'])) {
			$url .= '&customer_id=' . $config['customer_id'];
		}
		if(isset($config['status'])) {
			$url .= '&status=' . $config['status'];
		}
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$estimates = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $estimates : false;
	}


	/**
	 * Get estimate
	 *
	 * @param (int) estimate id
	 *
	 * @return (string) json string || false
	 */
	public function getEstimate($id, $config = array())
	{
		$url = $this->apiUrl . $this->estimatesUrl . $id . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		if(isset($config['accept'])) {
			$url .= '&accept=' . $config['accept'];
		}
		if(isset($config['print'])) {
			$url .= '&print=' . $config['print'];
		}
		
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$estimate = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $estimate : false;
	}
	
	/**
	 * Get all estimates
	 *
	 * @param (date) date start
	 * @param (date) date end
	 *
	 * @return (string) json string || false
	 */
	public function approveEstimates($config = array())
	{
		$url = $this->apiUrl . $this->estimatesUrl . $config['estimate_id'] .'/approve?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_POST, 1);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$response = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $response;
	}


	/**
	 * Get all customer Payments
	 *
	 * @param (date) date start
	 * @param (date) date end
	 *
	 * @return (string) json string || false
	 */
	public function allCustomerPayments($config = array())
	{
		$url = $this->apiUrl . $this->customerPaymentsUrl . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		if(isset($config['page'])) {
			$url .= '&page=' . $config['page'];
		}
		if(isset($config['customer_id'])) {
			$url .= '&customer_id=' . $config['customer_id'];
		}
		if(isset($config['filter_by'])) {
			$url .= '&filter_by=' . $config['filter_by'];
		}
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$customerPayments = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $customerPayments : false;
	}


	/**
	 * Get customer Payment
	 *
	 * @param (int) customer Payment id
	 *
	 * @return (string) json string || false
	 */
	public function getCustomerPayments($id)
	{
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $this->apiUrl . $this->customerPaymentsUrl . $id . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$customerPayment = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $customerPayment : false;
	}


	/**
	 * Create an invoice
	 *
	 * @param (string) json encoded
	 * @param (bool) send the invoice to the contact associated with the invoice
	 *
	 * @return (bool)
	 */
	public function postInvoice($invoice, $send = false)
	{
		$url = $this->apiUrl . $this->invoicesUrl;

		$data = array(
			'authtoken' 		=> $this->authtoken,
			'JSONString' 		=> $invoice,
			"organization_id" 	=> $this->organizationId
		);

		$ch = curl_init($url);

		curl_setopt_array($ch, array(
			CURLOPT_POST => 1,
			CURLOPT_POSTFIELDS => $data,
			CURLOPT_RETURNTRANSFER => true
		));

		$invoice = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 201 ? true : false;
	}


	/**
	 * Get all credit notes
	 *
	 * @param (date) date start
	 * @param (date) date end
	 *
	 * @return (string) json string || false
	 */
	public function allCreditNotes($config = array())
	{
		$url = $this->apiUrl . $this->creditnotesUrl . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		if(isset($config['date_start']) && isset($config['date_end'])) {
			$url .= '&date_start=' . $config['date_start'] . '&date_end=' . $config['date_end'];
		}
		if(isset($config['page'])) {
			$url .= '&page=' . $config['page'];
		}
		if(isset($config['customer_id'])) {
			$url .= '&customer_id=' . $config['customer_id'];
		}
		if(isset($config['filter_by'])) {
			$url .= '&filter_by=' . $config['filter_by'];
		}
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$creditnotes = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $creditnotes : false;
	}


	/**
	 * Get credit note
	 *
	 * @param (int) credit note id
	 *
	 * @return (string) json string || false
	 */
	public function getCreditNote($id, $config = array())
	{
		$url = $this->apiUrl . $this->creditnotesUrl . $id . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		if(isset($config['accept'])) {
			$url .= '&accept=' . $config['accept'];
		}
		if(isset($config['print'])) {
			$url .= '&print=' . $config['print'];
		}
		
		
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$creditnote = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $creditnote : false;
	}


	/**
	 * Create a credit note
	 *
	 * @param (string) json string
	 *
	 * @return (bool)
	 */
	public function postCreditNote($creditnote)
	{
		$url = $this->apiUrl . $this->creditnotesUrl;

		$data = array(
			'authtoken' 		=> $this->authtoken,
			'JSONString' 		=> $creditnote,
			"organization_id" 	=> $this->organizationId
		);

		$ch = curl_init($url);

		curl_setopt_array($ch, array(
			CURLOPT_POST => 1,
			CURLOPT_POSTFIELDS => $data,
			CURLOPT_RETURNTRANSFER => true
		));

		$creditnote = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 201 ? true : false;
	}


	/**
	 * Get HTTP code
	 */
	public function getHttpCode()
	{
		return $this->httpCode ? $this->httpCode : false;
	}


	/**
	 * Check API requests limit
	 *
	 */
	private function checkApiRequestsLimit()
	{
		$tempTime = time() - $this->startTime;
		if($this->apiRequestsCount >= $this->apiRequestsLimit && $tempTime < $this->apiTimeLimit) {
			usleep(($this->apiTimeLimit - $tempTime)*1000000);
			$this->apiRequestsCount = 1;
			$this->startTime = time();
		} else {
			$this->apiRequestsCount++;
		}
	}
	
	
	public function getStatementMailContent($config = array())
	{
		$url = $this->apiUrl . $this->contactsUrl . $config['customer_id'] . '/statements/email' . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		
		if(isset($config['start_date'])) {
			$url .= '&start_date=' . $config['start_date'];
		}
		if(isset($config['end_date'])) {
			$url .= '&end_date=' . $config['end_date'];
		}
		
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json"));
		$mailContent = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $mailContent : false;
	}
	
	public function postStatementMail($config, $mailData){
		$url = $this->apiUrl . $this->contactsUrl . $config['customer_id'] . '/statements/email' . '?authtoken=' . $this->authtoken . '&organization_id=' . $this->organizationId;
		
		if(isset($config['start_date'])) {
			$url .= '&start_date=' . $config['start_date'];
		}
		if(isset($config['end_date'])) {
			$url .= '&end_date=' . $config['end_date'];
		}

		$data = array(
			'authtoken' 		=> $this->authtoken,
			'JSONString' 		=> $mailData,
			"organization_id" 	=> $this->organizationId
		);

		$ch = curl_init($url);

		curl_setopt_array($ch, array(
			CURLOPT_POST => 1,
			CURLOPT_POSTFIELDS => $data,
			CURLOPT_RETURNTRANSFER => true
		));

		$response = curl_exec($ch);
		$this->httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);

		$this->checkApiRequestsLimit();

		return $this->httpCode == 200 ? $response : false;
	}
}